// Bike Rental Website - Version 4 (HTML/CSS/JS)

// Bike types data
const bikeTypes = [
  { 
    id: 'mountain', 
    name: 'Горный', 
    icon: 'icons/type-mountain.svg', 
    color: '#8B5CF6',
    description: 'Для горных трасс',
    price: 1500,
    model: 'Trek Mountain Pro'
  },
  { 
    id: 'city', 
    name: 'Городской', 
    icon: 'icons/type-city.svg', 
    color: '#7C3AED',
    description: 'Комфорт по городу',
    price: 1200,
    model: 'City Cruiser 2024'
  },
  { 
    id: 'kids', 
    name: 'Детский', 
    icon: 'icons/type-kids.svg', 
    color: '#A78BFA',
    description: 'Безопасно для детей',
    price: 800,
    model: 'Kids Bike Junior'
  },
  { 
    id: 'electric', 
    name: 'Электровелосипед', 
    icon: 'icons/type-electric.svg', 
    color: '#8B5CF6',
    description: 'С электромотором',
    price: 2000,
    model: 'Electric Speed X1'
  }
]

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'history'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initHistoryPage()
  }
  
  initSlider()
}

// Initialize history page
function initHistoryPage() {
  loadRentals()
  updateStats()
  
  // Tab switching
  const tabs = document.querySelectorAll('.tab-inline')
  tabs.forEach(tab => {
    tab.addEventListener('click', function() {
      tabs.forEach(t => t.classList.remove('active'))
      this.classList.add('active')
      const tabType = this.dataset.tab
      renderRentals(tabType)
    })
  })
  
  // Initial render
  renderRentals('active')
  
  // Poll for storage changes
  setInterval(() => {
    loadRentals()
    updateStats()
  }, 500)
}

// Load rentals from localStorage
function loadRentals() {
  const saved = localStorage.getItem('bikeRentals')
  let rentals = { active: [], past: [] }
  
  if (saved) {
    try {
      rentals = JSON.parse(saved)
    } catch (e) {
      rentals = { active: [], past: [] }
    }
  } else {
    // Default rentals
    rentals = {
      active: [
        {
          id: 1,
          model: 'Trek Mountain Pro',
          type: 'Горный',
          date: '2025-03-28',
          returnDate: '2025-04-05',
          price: 1500,
          status: 'active'
        },
        {
          id: 2,
          model: 'City Cruiser 2024',
          type: 'Городской',
          date: '2025-03-25',
          returnDate: '2025-04-02',
          price: 1200,
          status: 'active'
        }
      ],
      past: [
        {
          id: 3,
          model: 'Electric Speed X1',
          type: 'Электровелосипед',
          date: '2025-03-15',
          returnDate: '2025-03-20',
          price: 2000,
          review: 'Отличный велосипед, очень удобный!'
        },
        {
          id: 4,
          model: 'Kids Bike Junior',
          type: 'Детский',
          date: '2025-03-10',
          returnDate: '2025-03-12',
          price: 800,
          review: 'Ребенок в восторге!'
        }
      ]
    }
    localStorage.setItem('bikeRentals', JSON.stringify(rentals))
  }
  
  return rentals
}

// Update stats
function updateStats() {
  const rentals = loadRentals()
  const stats = [
    { 
      value: rentals.past.length, 
      label: 'Завершено',
      icon: 'icons/stat-bike.svg',
      color: '#8B5CF6'
    },
    { 
      value: rentals.past.reduce((sum, r) => sum + (r.price || 0), 0), 
      label: 'Потрачено',
      icon: 'icons/stat-dollar.svg',
      color: '#7C3AED'
    },
    { 
      value: rentals.past.filter(r => r.review).length, 
      label: 'Отзывов',
      icon: 'icons/stat-message.svg',
      color: '#A78BFA'
    }
  ]
  
  const statsCircles = document.getElementById('statsCircles')
  if (statsCircles) {
    statsCircles.innerHTML = stats.map(stat => `
      <div class="stat-circle">
        <div class="circle-icon" style="border-color: ${stat.color}40">
          <img src="${stat.icon}" alt="" width="24" height="24" />
        </div>
        <div class="circle-content">
          <div class="circle-value">
            ${stat.label === 'Потрачено' ? `${stat.value} ₽` : stat.value}
          </div>
          <div class="circle-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }

  // Update counts
  const activeCount = document.getElementById('activeCount')
  const pastCount = document.getElementById('pastCount')
  if (activeCount) activeCount.textContent = rentals.active.length
  if (pastCount) pastCount.textContent = rentals.past.length
}

// Render rentals
function renderRentals(type) {
  const rentals = loadRentals()
  const currentRentals = rentals[type]
  const timeline = document.getElementById('timeline')
  
  if (!timeline) return
  
  if (currentRentals.length === 0) {
    timeline.innerHTML = `
      <div class="empty-state">
        <img src="icons/rental-bike.svg" alt="" class="empty-icon" width="56" height="56" />
        <p>${type === 'active' ? 'У вас нет активных аренд' : 'История аренд пуста'}</p>
      </div>
    `
    return
  }
  
  timeline.innerHTML = currentRentals.map((rental, index) => {
    const startDate = new Date(rental.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })
    const endDate = new Date(rental.returnDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })
    
    return `
      <div class="timeline-item">
        <div class="timeline-marker ${type === 'past' ? 'past' : ''}"></div>
        <div class="timeline-content">
          <div class="timeline-header">
            <div class="timeline-bike">
              <img src="icons/rental-bike.svg" alt="" width="20" height="20" />
            </div>
            <div class="timeline-info">
              <h3 class="timeline-model">${rental.model}</h3>
              <p class="timeline-type">${rental.type}</p>
              ${rental.review ? `
                <div class="timeline-review">
                  <img src="icons/stat-message.svg" alt="" width="14" height="14" />
                  <span>${rental.review}</span>
                </div>
              ` : ''}
            </div>
            <div class="timeline-price">${rental.price} ₽</div>
          </div>
          <div class="timeline-dates">
            <span>${startDate} - ${endDate}</span>
            ${rental.status === 'active' ? '<span class="timeline-status">Активна</span>' : ''}
          </div>
        </div>
      </div>
    `
  }).join('')
}

// Initialize book page
function initBookPage() {
  renderBikeTypes()
  
  const form = document.getElementById('bookingForm')
  const returnDateInput = document.getElementById('returnDate')
  const reviewInput = document.getElementById('review')
  let selectedType = ''
  
  // Set date limits
  const today = new Date()
  const maxDate = new Date(today.getTime() + 30 * 24 * 60 * 60 * 1000)
  returnDateInput.min = today.toISOString().split('T')[0]
  returnDateInput.max = maxDate.toISOString().split('T')[0]
  
  // Type selection
  document.querySelectorAll('.type-list-item').forEach(item => {
    item.addEventListener('click', function() {
      document.querySelectorAll('.type-list-item').forEach(i => i.classList.remove('selected'))
      this.classList.add('selected')
      selectedType = this.dataset.type
      updatePriceSummary()
    })
  })
  
  // Date change
  returnDateInput.addEventListener('change', updatePriceSummary)
  
  // Form submit
  form.addEventListener('submit', function(e) {
    e.preventDefault()
    
    if (!selectedType || !returnDateInput.value) {
      alert('Пожалуйста, выберите тип велосипеда и дату возврата')
      return
    }
    
    const savedRentals = localStorage.getItem('bikeRentals')
    const rentals = savedRentals ? JSON.parse(savedRentals) : { active: [], past: [] }
    
    const bikeType = bikeTypes.find(t => t.id === selectedType)
    const newRental = {
      id: Date.now(),
      model: bikeType.model,
      type: bikeType.name,
      date: today.toISOString().split('T')[0],
      returnDate: returnDateInput.value,
      price: bikeType.price,
      status: 'active',
      review: reviewInput.value || null
    }
    
    rentals.active.push(newRental)
    localStorage.setItem('bikeRentals', JSON.stringify(rentals))
    
    // Show success message
    const successMsg = document.getElementById('successMessage')
    successMsg.style.display = 'flex'
    
    setTimeout(() => {
      window.location.href = 'index.html'
    }, 2000)
  })
}

// Render bike types
function renderBikeTypes() {
  const typesList = document.getElementById('typesList')
  if (!typesList) return
  
  typesList.innerHTML = bikeTypes.map(type => `
    <button type="button" class="type-list-item" data-type="${type.id}">
      <div class="type-list-icon" style="border-color: ${type.color}40">
        <img src="${type.icon}" alt="${type.name}" width="28" height="28" />
      </div>
      <div class="type-list-content">
        <h3 class="type-list-name">${type.name}</h3>
        <p class="type-list-desc">${type.description}</p>
      </div>
      <div class="type-list-price">${type.price} ₽/день</div>
    </button>
  `).join('')
}

// Update price summary
function updatePriceSummary() {
  const selectedItem = document.querySelector('.type-list-item.selected')
  const returnDateInput = document.getElementById('returnDate')
  const priceSummary = document.getElementById('priceSummary')
  
  if (!selectedItem || !returnDateInput.value || !priceSummary) return
  
  const typeId = selectedItem.dataset.type
  const bikeType = bikeTypes.find(t => t.id === typeId)
  if (!bikeType) return
  
  const today = new Date()
  const returnDate = new Date(returnDateInput.value)
  const days = Math.ceil((returnDate - today) / (1000 * 60 * 60 * 24))
  const total = bikeType.price * days
  
  document.getElementById('summaryPrice').textContent = `${bikeType.price} ₽`
  document.getElementById('summaryDays').textContent = days
  document.getElementById('totalPrice').textContent = `${total} ₽`
  
  priceSummary.style.display = 'block'
}

// Initialize slider
function initSlider() {
  const slider = document.getElementById('slider')
  if (!slider) return
  
  const slides = slider.querySelectorAll('.slide')
  const dots = slider.querySelectorAll('.dot')
  const prevBtn = document.getElementById('sliderPrev')
  const nextBtn = document.getElementById('sliderNext')
  
  let currentIndex = 0
  
  function showSlide(index) {
    slides.forEach((slide, i) => {
      slide.classList.toggle('active', i === index)
    })
    dots.forEach((dot, i) => {
      dot.classList.toggle('active', i === index)
    })
  }
  
  function nextSlide() {
    currentIndex = (currentIndex + 1) % slides.length
    showSlide(currentIndex)
  }
  
  function prevSlide() {
    currentIndex = (currentIndex - 1 + slides.length) % slides.length
    showSlide(currentIndex)
  }
  
  if (prevBtn) prevBtn.addEventListener('click', prevSlide)
  if (nextBtn) nextBtn.addEventListener('click', nextSlide)
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      currentIndex = index
      showSlide(currentIndex)
    })
  })
  
  // Auto-advance every 3 seconds
  setInterval(nextSlide, 3000)
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

